"""Module that defines the bare Grassmann operations."""

import numpy as np

def log_alt(c: np.ndarray, c0: np.ndarray) -> np.ndarray:
    """Grassmann logarithm alterative version."""
    c0c_inv = np.linalg.inv(c0.T @ c)
    l = c @ c0c_inv - c0
    q, s, vt = np.linalg.svd(l, full_matrices=False)
    arctan_s = np.diag(np.arctan(s))
    return q @ arctan_s @ vt

def log(c: np.ndarray, c0: np.ndarray) -> np.ndarray:
    """Grassmann logarithm."""
    psi, s, rt = np.linalg.svd(c.T @ c0, full_matrices=False)
    cstar = c @ psi @ rt
    l = (np.identity(c.shape[0]) - c0 @ c0.T) @ cstar
    u, s, vt = np.linalg.svd(l, full_matrices=False)
    arcsin_s = np.diag(np.arcsin(s))
    return u @ arcsin_s @ vt

def exp(gamma: np.ndarray, c0: np.ndarray) -> np.ndarray:
    """Grassmann exponential."""
    q, s, vt = np.linalg.svd(gamma, full_matrices=False)
    sin_s = np.diag(np.sin(s))
    cos_s = np.diag(np.cos(s))
    return c0 @ vt.T @ cos_s @ vt + q @ sin_s @ vt
