"""Module which provides functionality to perform fitting."""

import abc
from typing import List
import numpy as np

class AbstractFitting(abc.ABC):

    """Base class for fitting schemes."""

    def __init__(self, **kwargs):
        self.set_options(**kwargs)

    @abc.abstractmethod
    def set_options(self, **kwargs):
        """Base method for setting options."""

    @abc.abstractmethod
    def compute(self, vectors: List[np.ndarray], target:np.ndarray):
        """Base method for computing new fitting coefficients."""

    def linear_combination(self, vectors: List[np.ndarray],
            coefficients: np. ndarray) -> np.ndarray:
        """Given a set of vectors (or matrices) and the corresponding
        coefficients, build their linear combination."""
        result = np.zeros(vectors[0].shape, dtype=np.float64)
        for coeff, vector in zip(coefficients, vectors):
            result += vector*coeff
        return result

class LeastSquare(AbstractFitting):

    """Simple least square minimization fitting."""

    supported_options = {
        "regularization": 0.0,
    }

    def set_options(self, **kwargs):
        """Set options for least square minimization"""
        self.options = {}
        for key, value in kwargs.items():
            if key in self.supported_options:
                self.options[key] = value
            else:
                raise ValueError(f"Unsupported option: {key}")

        for option, default_value in self.supported_options.items():
            if option not in self.options:
                self.options[option] = default_value

        if self.options["regularization"] < 0 \
                or self.options["regularization"] > 100:
            raise ValueError("Unsupported value for regularization")

    def compute(self, vectors: List[np.ndarray], target: np.ndarray):
        """Given a set of vectors and a target return the fitting
        coefficients."""
        matrix = np.vstack(vectors).T
        coefficients, _, _, _ = np.linalg.lstsq(matrix, target, rcond=None)
        return np.array(coefficients, dtype=np.float64)

class QuasiTimeReversible(AbstractFitting):

    """Quasi time reversible fitting scheme. Not yet implemented."""

    def set_options(self, **kwargs):
        """Set options for quasi time reversible fitting"""

    def compute(self, vectors: List[np.ndarray], target: np.ndarray):
        """Time reversible least square minimization fitting."""
