"""Module that defines a circular buffer for storing the last properties
in a molecular dynamics simulation."""

from typing import List, Tuple
import numpy as np

class CircularBuffer:

    """Circular buffer to store the last `n` matrices."""

    def __init__(self, n: int, shape: Tuple[int, ...]):
        self.n = n
        self.shape = shape
        self.buffer = [np.zeros(shape, dtype=np.float64) for _ in range(n)]
        self.index = 0
        self.count = 0

    def push(self, data):
        """Add a new matrix to the buffer."""
        self.buffer[self.index] = data.copy()
        self.index = (self.index + 1) % self.n
        if self.count < self.n:
            self.count += 1

    def get(self, m) -> List[np.ndarray]:
        """Get the last `m` matrices."""
        if m < 0:
            raise ValueError("`m` should be larger than 0.")
        if m > self.n:
            raise ValueError("`m` should be less than or equal to the buffer `n`.")
        if m > self.count:
            raise ValueError("`m` is larger than the stored matrices.")

        start_idx = (self.index - m) % self.n
        return [self.buffer[(start_idx + i) % self.n] for i in range(m)]
